//Check HTC Android 2 built-in browser
var isAndroidHTC = navigator.userAgent.match(/HTC/i) && !navigator.userAgent.match(/Android 4/i);

/**
 * Object definition for handling keyboard events for Crossword
 */
var CrosswordInput = $.extend(PuzzleInput, {
    /**
     * Returns true if a valid crossword input key is pressed (Letters)
     */
    isValidInput: function(event) {
        return ((event.keyCode >= 65 && event.keyCode <= 90) || (event.keyCode >= 97 && event.keyCode <= 122)
                || (event.which >= 65 && event.which <= 90) || (event.which >= 97 && event.which <= 122));
    }
});

var CrosswordFlags = {
    HINT: 1,
    SOLUTION: 2,
    HINT_WORD: 4,
    INCREASE_FONT: 8,
    COMPETITION_UPON_COMPLETION: 16,
    COMPETITION_BEFORE_COMPLETION: 32,
};

/**
 * Object definition for crossword
 */
var Crossword = $.extend(Puzzle, {
    solutionTimeout: -1,
    /**
     * Initializes components and event listeners for crossword
     * Currently, clue clicks, cell clicks and key inputs are
     * being listened by this class.
     */
    init: function(options) {
        //Get options
        $.extend(this.defaults, options);

        //Insert input field
        this.insertUserInputField();

        //Set parent objects
        this.parentObjects.grid = $(this.defaults.gridId);
        this.parentObjects.clues = $(this.defaults.cluesId);

        /**
         * CLUE CLICKS: Deselect all clues, remove highlight.
         * Then, highlight clicked clue, select it logically
         * by assigning to class property.
         */
        $(this.defaults.cluesId).click(function() {
            //Set event source
            Crossword.cellEventSource = EventSource.ClueClick;
            //Deselect old clues
            Crossword.deselectClues();
            //Deselect old cells
            Crossword.deselectCells();
            //Mark selected clue
            Crossword.highlightClickedClue(this);
            //Select clue logically & trigger clue tracking
            Crossword.selectClue(this, true);
            //Select answer slot for clue
            Crossword.selectCellForClue(this, 0);
        });

        //CELL CLICKS
        $(this.defaults.cellsId).click(function(event) {
            if (!$(this).hasClass(Crossword.defaults.blankClass))
            {                
                //Determine if the source of event is user click
                if (event.originalEvent)
                    Crossword.cellEventSource = EventSource.CellClick;

                //Deselect old clues
                if (Crossword.shouldSelectedCluesChange())
                    Crossword.deselectClues();

                //Deselect old cells
                Crossword.deselectCells();
                //Select cell logically
                Crossword.selectCluesForCell(this);
                //Mark input cell
                Crossword.hightlightInputCell(this);
                //Reposition user input field according to cell
                Crossword.repositionUserInputField();
                //Enable user input
                $(Crossword.defaults.userInputId).focus();

                //Update selected group index
                if (Crossword.shouldAnswerSlotChange()) {
                    Crossword.lastSelected.cellGroupIndex = Crossword.findCellIndexInGroup(Crossword.lastSelected.cell, Crossword.lastSelected.cellGroup);
                }
            }
            else {
                event.preventDefault();
                return false;
            }
        });

        var androidChrome = (navigator.userAgent.toLowerCase().indexOf("android") > -1 && navigator.userAgent.toLowerCase().indexOf("chrome") > -1);

        if (androidChrome)
        {
                    //Backspace and arrow keys should be handled with "keydown"
            $(this.defaults.userInputId).keyup(function(event) {
                //Get user input
                var inputLastChar = CrosswordInput.getUserInput(event, Crossword.defaults.userInputId);

                if (CrosswordInput.isBackspace(event))
                {
                    //Backspace
                    //Set cell text to empty value
                    Crossword.setCellValue(Crossword.lastSelected.cell, "");
                    //Focus previous cell
                    Crossword.focusPrevInputCell();
                }

                if (CrosswordInput.isValidInput(event))
                {
                    //Set cell text to inputted value
                    Crossword.setCellValue(Crossword.lastSelected.cell, inputLastChar);
                    //Focus next cell
                    Crossword.focusNextInputCell();
                }
                else
                {
                    //console.log("Key is not valid!");
                }
            });
        }
        else
        {
            //Backspace and arrow keys should be handled with "keydown"
            $(this.defaults.userInputId).keydown(function(event) {
                //Get user input
                CrosswordInput.getUserInput(event, Crossword.defaults.userInputId);

                if (CrosswordInput.isBackspace(event))
                {
                    //Backspace
                    //Set cell text to empty value
                    Crossword.setCellValue(Crossword.lastSelected.cell, "");
                    //Focus previous cell
                    Crossword.focusPrevInputCell();
                }
                else if (CrosswordInput.isUp(event))
                {
                    //Up arrow
                    Crossword.focusCellAtDirection(Crossword.lastSelected.cell, PuzzleDirection.UP);
                }
                else if (CrosswordInput.isRight(event))
                {
                    //Right arrow
                    Crossword.focusCellAtDirection(Crossword.lastSelected.cell, PuzzleDirection.RIGHT);
                }
                else if (CrosswordInput.isDown(event))
                {
                    //Down arrow
                    Crossword.focusCellAtDirection(Crossword.lastSelected.cell, PuzzleDirection.DOWN);
                }
                else if (CrosswordInput.isLeft(event))
                {
                    //Left arrow
                    Crossword.focusCellAtDirection(Crossword.lastSelected.cell, PuzzleDirection.LEFT);
                }
                else
                {
                    //console.log("Key is not valid!");
                }
            });

            //USER INPUT should be handled with "keypress"
            $(this.defaults.userInputId).keypress(function(event) {
                //Get user input
                var inputLastChar = CrosswordInput.getUserInput(event, Crossword.defaults.userInputId);

                if (CrosswordInput.isValidInput(event))
                {
                    //Set cell text to inputted value
                    Crossword.setCellValue(Crossword.lastSelected.cell, inputLastChar);
                    //Focus next cell
                    Crossword.focusNextInputCell();
                }
                else
                {
                    //console.log("Key is not valid!");
                }
            });
        }

        //Font size toggle for clues
        $(this.defaults.fontSizeButton).click(function() {
            if ($(Crossword.defaults.cluesId).hasClass(Crossword.defaults.fontSizeLargeClass))
                $(Crossword.defaults.cluesId).removeClass(Crossword.defaults.fontSizeLargeClass);
            else
                $(Crossword.defaults.cluesId).addClass(Crossword.defaults.fontSizeLargeClass);
            
            TrackCs.flags |= CrosswordFlags.INCREASE_FONT;
        });

        //Zoom buttons
        $(this.defaults.zoomInButton).click(function() {
            Puzzle.zoomIn();
        });

        $(this.defaults.zoomOutButton).click(function() {
            Puzzle.zoomOut();
        });
    },
    /**
     * Inserts an input field to web page for user input
     */
    insertUserInputField: function() {
        var userInput = document.createElement("input");
        userInput.id = this.defaults.userInputId.replace("#", "");
        userInput.type = "text";
        userInput.maxlength = "1";
        userInput.spellcheck = "false";
        userInput.style.top = "-10000px";
        userInput.style.width = "10px";
        userInput.style.height = "10px";
        userInput.style.fontSize = "1px";
        userInput.style.textTransform = "uppercase";
        userInput.style.position = "absolute";
        userInput.style.zIndex = "-9999";
        userInput.style.fontSize = "10px";
        document.body.innerHTML = userInput.outerHTML + document.body.innerHTML;
    },
    repositionUserInputField: function() {
        //Scroll to input cell
        if (/*typeof(isClientMobile) !== undefined && isClientMobile &&*/ this.lastSelected.cell) {
            $(this.defaults.userInputId).css("top", $(this.lastSelected.cell).offset().top);
            $(this.defaults.userInputId).css("left", $(this.lastSelected.cell).offset().left);
        }
    },
    /**
     * Invokes cell click event programatically
     */
    invokeCellClick: function(cell, eventSource) {
        this.cellEventSource = eventSource;
        $(cell).click();
    },
    /**
     * Determines if answer slot should change during the event, depending on the event source
     */
    shouldAnswerSlotChange: function() {
        if (this.cellEventSource == EventSource.CellClick || this.cellEventSource == EventSource.ClueClick)
            return true;

        return false;
    },
    /**
     * Highlights the cell that takes user input
     */
    hightlightInputCell: function(cell) {
        $(cell).addClass(this.defaults.selectedClass);
        this.lastSelected.cell = cell;
    },
    /**
     * Removes the highlight from selected cell & answer slot
     */
    deselectCells: function() {
        //Remove highlight from input cell
        if (this.lastSelected.cell != undefined)
        {
            $(this.lastSelected.cell).removeClass(this.defaults.selectedClass);
            this.lastSelected.cell = undefined;
        }

        //If this is caused by a mouse event, remove highlight from answer slot
        if (this.shouldAnswerSlotChange())
        {
            for (var i = 0; i < this.lastSelected.cellGroup.length; i++)
            {
                var cellX = this.lastSelected.cellGroup[i][0];
                var cellY = this.lastSelected.cellGroup[i][1];
                $("#cell-" + cellX + "-" + cellY).removeClass(this.defaults.selectedGroupClass);
            }

            //Reset selected group
            this.lastSelected.cellGroup = new Array();
        }
    },
    shouldSelectClue: function() {
        return (this.cellEventSource == EventSource.CellClick);
    },
    /**
     * Selects the related clues. For cells on the intersection of two answers, two clues will be selected.
     * Clues can also have related clues, so they will also be highlighted.
     */
    selectCluesForCell: function(cell) {
        //Get clues of the selected cell
        var clueIds = $(cell).data("clueid").toString().split("-");

        //If this is performed on a cell that is already reverted,
        //then take array in natural order. Otherwise revert the array, and mark the cell as reverted.
        if ($(cell).data("reverted") == 1)
        {
            $(cell).data("reverted", 0);
        }
        else
        {
            $(cell).data("reverted", 1);
            clueIds.reverse();
        }

        //One answer slot will be selected
        var asnwerSlotSelected = false;
        //Cells might be linked to more than one clue.
        //For each clue, highlight clue element & select answer slot
        for (var i = 0; i < clueIds.length; i++)
        {
            //Next clue selector
            var clue = $("#clue-" + clueIds[i]);

            if (this.cellEventSource == EventSource.ClueClick) {
                //If a clue is clicked, limit cell selection with clue direction
                var dirs = this.getClueProperty(clue, "direction");
                var lastSelectedDirs = this.getClueProperty(Crossword.lastSelected.clue, "direction");

                var correctClue = dirs.length == lastSelectedDirs.length;
                var length = Math.min(dirs.length, lastSelectedDirs.length);
                
                for(var ii = 0, ii_n = length; ii < ii_n && correctClue; ++ii)
                {
                    correctClue = dirs[ii] == lastSelectedDirs[ii];
                }
                
                if (correctClue) 
                {
                    //Select clue, and trigger clue tracking only for the first round of the loop
                    if (Crossword.shouldSelectClue())
                    {
                        this.selectClue(clue, (i == 0));
                    }

                    //Select answer slot
                    if (!asnwerSlotSelected && this.selectAnswerSlot(clue, 0))
                    {
                        asnwerSlotSelected = true;
                    }
                    break;
                }
            }
            else {
                //Select clue, and trigger clue tracking only for the first round of the loop
                if (Crossword.shouldSelectClue())
                    this.selectClue(clue, (i == 0));

                //Select answer slot
                if (!asnwerSlotSelected && this.selectAnswerSlot(clue, 0))
                    asnwerSlotSelected = true;
            }
        }
    },
    /**
     * Highlights ad logically selected the answer slot for user input.
     * Selected cells are stored in cell group variable, so they can be accessed for any operation.
     */
    selectAnswerSlot: function(clue, index) {
        //If the source is not a mouse event, then don't change answer slot
        if (!this.shouldAnswerSlotChange())
            return true;

        //Get all cell coordinates for clue answer
        var x1 = this.getClueProperty(clue, "x1");
        var x2 = this.getClueProperty(clue, "x2");
        var y1 = this.getClueProperty(clue, "y1");
        var y2 = this.getClueProperty(clue, "y2");
        var dirs = this.getClueProperty(clue, "direction");

        //Check if all the array have the same size
        if (x1.length > 0 && x2.length == x1.length && y1.length == x1.length && y2.length == x1.length && dirs.length == x1.length)
        {
            if (index >= x1.length)
                index = x1.length - 1;

            for (; index < x1.length; index++)
            {
                var xStr = parseInt(x1[index]);
                var xEnd = parseInt(x2[index]);
                var yStr = parseInt(y1[index]);
                var yEnd = parseInt(y2[index]);

                if (dirs[index] == "across")
                {
                    for (var xCor = xStr; xCor <= xEnd; xCor++)
                    {
                        $("#cell-" + xCor + "-" + yStr).addClass(this.defaults.selectedGroupClass);
                        this.lastSelected.cellGroup.push(new Array(xCor, yStr));
                    }
                }
                else if (dirs[index] == "down")
                {
                    for (var yCor = yStr; yCor <= yEnd; yCor++)
                    {
                        $("#cell-" + xStr + "-" + yCor).addClass(this.defaults.selectedGroupClass);
                        this.lastSelected.cellGroup.push(new Array(xStr, yCor));
                    }
                }
            }

            return true;
        }
        else
        {
            //console.log("Answer slots are not defined properly for this clue");
        }

        return false;
    },
    /**
     * Determines if clue selection should change with user input (Click, Type etc.)
     */
    shouldSelectedCluesChange: function() {
        if (this.cellEventSource == EventSource.ClueClick)
            return false;

        return true;
    },
    /**
     * Highlights a clue and adds it to selected group
     */
    highlightClue: function(clue) {
        $(clue).addClass(this.defaults.selectedGroupClass);
        this.lastSelected.clueGroup.push(clue);

        //Scroll to clue
        this.scrollToClue(clue);
    },
    /**
     * Highlights clicked clue and stores it in last selected clue
     */
    highlightClickedClue: function(clue) {
        $(this.lastSelected.clue).removeClass(this.defaults.selectedClass);
        $(clue).addClass(this.defaults.selectedClass);
        this.lastSelected.clue = clue;
    },
    /**
     * Removes highlight and deselects clues
     */
    deselectClues: function() {
        //Deselect active clue group only if answer slot changes
        if (this.shouldAnswerSlotChange()) {
            $(this.lastSelected.clue).removeClass(this.defaults.selectedClass);

            for (var i = 0; i < this.lastSelected.clueGroup.length; i++)
                $(this.lastSelected.clueGroup[i]).removeClass(this.defaults.selectedGroupClass);

            this.lastSelected.clueGroup = new Array();
        }
    },
    /**
     * Highlight a clue and all the other clues related to it
     */
    selectClue: function(clue, shouldTriggerTracking) {
        this.highlightClue(clue);

        //Do tracking only if clue change was caused by click event and answer slot has changed
        if (shouldTriggerTracking && this.shouldAnswerSlotChange())
            TrackCs.saveClueStartTime(clue);

        if ($(clue).data("islink") == 1)
        {
            var linkId = $(clue).data("linkid");
            this.highlightClue($("#" + linkId));
        }
    },
    /**
     * Determines if scroll position should change during the event, depending on the event source
     */
    shouldScrollToClue: function() {
        if (this.cellEventSource == EventSource.CellClick || this.cellEventSource == EventSource.ClueClick)
            return true;

        return false;
    },
    /**
     * Sets the scroll position to the clue
     */
    scrollToClue: function(clue) {
        //If the source is not a mouse event, then don't change scroll position
        if (!this.shouldScrollToClue())
            return;

        if ($(clue).parents(".clues-across").length > 0)
            acrossScroll.doScrollTop($(clue).position().top);
        else if ($(clue).parents(".clues-down").length > 0)
            downScroll.doScrollTop($(clue).position().top);
    },
    /**
     * Gets a data attribute of a clue element
     */
    getClueProperty: function(clue, cId) {
        return $(clue).data(cId).toString().split("-");
    },
    /**
     * Selects the cells for a clue by comparing if the cell is primary for clue (its positions are defined in the clue).
     * Click event for the selected cell is invoked so the coloring & selection is performed.
     */
    selectCellForClue: function(clue, index) {
        //Get all cell coordinates for clue answer
        var x1 = this.getClueProperty(clue, "x1");
        var y1 = this.getClueProperty(clue, "y1");

        //Check if all the array have the same size
        if (x1.length > 0 && y1.length == x1.length)
        {
            if (index >= x1.length)
                index = x1.length - 1;

            var cellX = parseInt(x1[index]);
            var cellY = parseInt(y1[index]);
            this.invokeCellClick($("#cell-" + cellX + "-" + cellY), EventSource.ClueClick);
        }
        else
        {
            //console.log("Answer slots are not defined properly for this clue");
        }
    },
    /**
     * Sets the value of a cell by setting inner HTML
     */
    setCellValue: function(cell, val) {
        $(cell).children(Crossword.defaults.cellInputId).html(val);
        $(cell).removeClass(this.defaults.invalidCellClass);

        if(val != ''){
            Crossword.createSolutionTimeout();} //createSolutionTimeout(Crossword, Crossword.solutionTimeout);}
    },
    /**
     * Finds the indice of a given cell in the given cell group
     */
    findCellIndexInGroup: function(cell, cellGroup, minValue) {
        if (!minValue)
            minValue = 0;

        //Get cell coordinates
        var cellDetails = cell.id.toString().split("-");
        var cellX = parseInt(cellDetails[1]);
        var cellY = parseInt(cellDetails[2]);

        //Find selected cell in the group
        for (var index = 0; index < cellGroup.length; index++)
        {
            var curX = this.lastSelected.cellGroup[index][0];
            var curY = this.lastSelected.cellGroup[index][1];

            if (index >= minValue && cellX == curX && cellY == curY) {
                return index;
            }
        }

        return -1;
    },
    /**
     * Focuses next cell for user input. This function is used when user fills in an answer slot.
     */
    focusNextInputCell: function() {
        //Check if current cell is defined
        if (this.lastSelected.cell != undefined)
        {
            //To make while loop fail-safe, we limit try count
            var tryCount = 2;

            do {
                tryCount--;

                //Get cell coordinates
                var index = this.findCellIndexInGroup(this.lastSelected.cell,
                        this.lastSelected.cellGroup,
                        this.lastSelected.cellGroupIndex);
                index++;

                if (index < this.lastSelected.cellGroup.length)
                {
                    var nextCellId = "#cell-" + this.lastSelected.cellGroup[index][0]
                            + "-" + this.lastSelected.cellGroup[index][1];
                    this.invokeCellClick(nextCellId, EventSource.KeyInput);
                    Crossword.lastSelected.cellGroupIndex++;
                }
            }
            while (index < this.lastSelected.cellGroupIndex && tryCount > 0);
        }
    },
    /**
     * Focuses next cell for user input. This function is used when user deletes an answer slot with backspace.
     */
    focusPrevInputCell: function() {
        //Check if current cell is defined
        if (this.lastSelected.cell != undefined)
        {
            //To make while loop fail-safe, we limit try count
            var tryCount = 2;

            do {
                tryCount--;

                //Get cell coordinates
                var index = this.findCellIndexInGroup(this.lastSelected.cell,
                        this.lastSelected.cellGroup,
                        this.lastSelected.cellGroupIndex);
                index--;

                if (index >= 0)
                {
                    var prevCellId = "#cell-" + this.lastSelected.cellGroup[index][0]
                            + "-" + this.lastSelected.cellGroup[index][1];
                    this.invokeCellClick(prevCellId, EventSource.KeyInput);
                    Crossword.lastSelected.cellGroupIndex--;
                }
            }
            while (index < this.lastSelected.cellGroupIndex && tryCount > 0);
        }
    },
    /**
     * Focuses next cell for user input. This function is used when user fills in an answer slot.
     */
    focusCellAtDirection: function(cell, direction) {
        //Get cell coordinates
        var cellDetails = $(cell).attr("id").toString().split("-");
        var cellX = parseInt(cellDetails[1]);
        var cellY = parseInt(cellDetails[2]);

        //Determine the new cell coordinates based on given direction
        if (direction == PuzzleDirection.UP)
            cellY = ((cellY - 1) <= 0) ? Puzzle.size.height : (cellY - 1);
        else if (direction == PuzzleDirection.RIGHT)
            cellX = ((cellX + 1) > Puzzle.size.width) ? 1 : (cellX + 1);
        else if (direction == PuzzleDirection.DOWN)
            cellY = ((cellY + 1) > Puzzle.size.height) ? 1 : (cellY + 1);
        else if (direction == PuzzleDirection.LEFT)
            cellX = ((cellX - 1) <= 0) ? Puzzle.size.width : (cellX - 1);

        var nextCellId = "#cell-" + cellX + "-" + cellY;
        var nextCell = $(nextCellId);

        //If neighbour cell is blank, try next
        if (nextCell.hasClass(Crossword.defaults.blankClass)) {
            Crossword.focusCellAtDirection(nextCell, direction);
            return;
        }

        this.invokeCellClick(nextCellId, EventSource.CellClick);
    },
    moveToNextHintCell: function() {
        var cellX = 1;
        var cellY = 1;

        //Get cell coordinates
        if (Puzzle.lastSelected.cellGroup && Puzzle.lastSelected.cell) {
            //Find cell in the group
            var selectedIndex = Crossword.findCellIndexInGroup(Puzzle.lastSelected.cell, Puzzle.lastSelected.cellGroup);
            if (selectedIndex < 0)
                selectedIndex = 0;
            else if (selectedIndex + 1 < Puzzle.lastSelected.cellGroup.length)
                selectedIndex++;

            cellX = parseInt(Puzzle.lastSelected.cellGroup[selectedIndex][0]);
            cellY = parseInt(Puzzle.lastSelected.cellGroup[selectedIndex][1]);
        }

        var nextCellId = "#cell-" + cellX + "-" + cellY;
        this.invokeCellClick(nextCellId, EventSource.KeyInput);
    },
    fixHeightHTCAndroid: function()
    {
        if (isAndroidHTC) {
            setTimeout(function() {
                var heightRequired = $("#puzzle-grid").height();
                var cluesAcross = $($(".clues")[0]);

                if (cluesAcross.css("float") === "none")
                    heightRequired += cluesAcross.height() * 2;
                
                if ($(".banner-image").length > 0)
                    heightRequired += $(".banner-image").height();

                //Add keyboard height
                heightRequired += 300;
                //Set body height
                $("body").height(heightRequired);
            }, 1000);
        }
    }
});